'use client'

import { useState, useEffect } from 'react'
import { useParams, useRouter } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Cliente, Suscripcion, Producto } from '@/lib/types'
import Link from 'next/link'

export default function EditarPagoPage() {
  const params = useParams()
  const router = useRouter()
  const [loading, setLoading] = useState(true)
  const [guardando, setGuardando] = useState(false)
  const [clientes, setClientes] = useState<Cliente[]>([])
  const [suscripciones, setSuscripciones] = useState<Suscripcion[]>([])
  const [productos, setProductos] = useState<Producto[]>([])
  const [formData, setFormData] = useState({
    cliente_id: '',
    suscripcion_id: '',
    producto_id: '',
    monto: '',
    fecha_pago: '',
    fecha_inicio: '',
    fecha_fin: '',
    metodo_pago: '',
    referencia: '',
    estado: 'pendiente' as 'pendiente' | 'completado' | 'cancelado',
    incluir_iva: false,
    notas: '',
  })

  useEffect(() => {
    if (params.id) {
      fetchPago(params.id as string)
      fetchClientes()
      fetchProductos()
    }
  }, [params.id])

  useEffect(() => {
    if (formData.cliente_id) {
      fetchSuscripciones(formData.cliente_id)
    } else {
      setSuscripciones([])
    }
  }, [formData.cliente_id])

  async function fetchPago(id: string) {
    try {
      const { data, error } = await supabase
        .from('pagos')
        .select('*')
        .eq('id', id)
        .single()

      if (error) throw error
      
      const fechaInicio = data.fecha_inicio ? data.fecha_inicio.split('T')[0] : data.fecha_pago.split('T')[0]
      const fechaFin = data.fecha_fin ? data.fecha_fin.split('T')[0] : ''
      
      setFormData({
        cliente_id: data.cliente_id,
        suscripcion_id: data.suscripcion_id || '',
        producto_id: data.producto_id || '',
        monto: data.monto.toString(),
        fecha_pago: data.fecha_pago.split('T')[0],
        fecha_inicio: fechaInicio,
        fecha_fin: fechaFin,
        metodo_pago: data.metodo_pago || '',
        referencia: data.referencia || '',
        estado: data.estado,
        incluir_iva: data.incluir_iva || false,
        notas: data.notas || '',
      })

      if (data.cliente_id) {
        await fetchSuscripciones(data.cliente_id)
      }
    } catch (error) {
      console.error('Error fetching pago:', error)
    } finally {
      setLoading(false)
    }
  }

  async function fetchClientes() {
    try {
      const { data, error } = await supabase
        .from('clientes')
        .select('*')
        .eq('activo', true)
        .order('nombre')

      if (error) throw error
      setClientes(data || [])
    } catch (error) {
      console.error('Error fetching clientes:', error)
    }
  }

  async function fetchSuscripciones(clienteId: string) {
    try {
      const { data, error } = await supabase
        .from('suscripciones')
        .select('*')
        .eq('cliente_id', clienteId)
        .order('created_at', { ascending: false })

      if (error) throw error
      setSuscripciones(data || [])
    } catch (error) {
      console.error('Error fetching suscripciones:', error)
    }
  }

  async function fetchProductos() {
    try {
      const { data, error } = await supabase
        .from('productos')
        .select('*')
        .eq('activo', true)
        .order('nombre')

      if (error) throw error
      setProductos(data || [])
    } catch (error) {
      console.error('Error fetching productos:', error)
    }
  }

  function handleProductoChange(productoId: string) {
    const producto = productos.find(p => p.id === productoId)
    if (producto) {
      setFormData({
        ...formData,
        producto_id: productoId,
        monto: producto.precio.toString(),
      })
    } else {
      setFormData({
        ...formData,
        producto_id: productoId,
      })
    }
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()
    setGuardando(true)

    try {
      const { error } = await supabase
        .from('pagos')
        .update({
          cliente_id: formData.cliente_id,
          suscripcion_id: formData.suscripcion_id || null,
          producto_id: formData.producto_id || null,
          monto: parseFloat(formData.monto),
          fecha_pago: formData.fecha_pago,
          fecha_inicio: formData.fecha_inicio || formData.fecha_pago || null,
          fecha_fin: formData.fecha_fin || null,
          metodo_pago: formData.metodo_pago || null,
          referencia: formData.referencia || null,
          estado: formData.estado,
          incluir_iva: formData.incluir_iva,
          notas: formData.notas || null,
        })
        .eq('id', params.id)

      if (error) throw error

      router.push(`/pagos/${params.id}`)
    } catch (error: any) {
      console.error('Error updating pago:', error)
      alert(`Error al actualizar el pago: ${error.message}`)
    } finally {
      setGuardando(false)
    }
  }

  if (loading) {
    return <div className="text-center p-8">Cargando...</div>
  }

  return (
    <div className="max-w-2xl mx-auto">
      <div className="mb-4">
        <Link href={`/pagos/${params.id}`} className="text-blue-600 hover:underline">← Volver</Link>
      </div>
      <h1 className="text-3xl font-bold mb-6">Editar Pago</h1>
      <form onSubmit={handleSubmit} className="bg-white p-6 rounded-lg shadow space-y-4">
        <div>
          <label className="block text-sm font-medium mb-1">Cliente *</label>
          <select
            required
            value={formData.cliente_id}
            onChange={(e) => setFormData({ ...formData, cliente_id: e.target.value, suscripcion_id: '' })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona un cliente</option>
            {clientes.map((cliente) => (
              <option key={cliente.id} value={cliente.id}>
                {cliente.nombre || cliente.empresa || 'Cliente sin nombre'} {cliente.nombre && cliente.empresa ? `- ${cliente.empresa}` : ''}
              </option>
            ))}
          </select>
        </div>

        {formData.cliente_id && suscripciones.length > 0 && (
          <div>
            <label className="block text-sm font-medium mb-1">Suscripción (opcional)</label>
            <select
              value={formData.suscripcion_id}
              onChange={(e) => setFormData({ ...formData, suscripcion_id: e.target.value })}
              className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            >
              <option value="">Sin suscripción</option>
              {suscripciones.map((suscripcion) => (
                <option key={suscripcion.id} value={suscripcion.id}>
                  {suscripcion.tipo === 'anual' ? 'Anual' : 'Mensual'} - ${suscripcion.precio.toLocaleString()}
                </option>
              ))}
            </select>
          </div>
        )}

        <div>
          <label className="block text-sm font-medium mb-1">Producto o Servicio (opcional)</label>
          <select
            value={formData.producto_id}
            onChange={(e) => handleProductoChange(e.target.value)}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona un producto o servicio</option>
            {productos.map((producto) => (
              <option key={producto.id} value={producto.id}>
                {producto.nombre} - ${producto.precio.toLocaleString()}
              </option>
            ))}
          </select>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Monto *</label>
          <input
            type="number"
            required
            step="0.01"
            min="0"
            value={formData.monto}
            onChange={(e) => setFormData({ ...formData, monto: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="0.00"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Fecha de Pago *</label>
          <input
            type="date"
            required
            value={formData.fecha_pago}
            onChange={(e) => setFormData({ ...formData, fecha_pago: e.target.value, fecha_inicio: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Fecha de Inicio</label>
          <input
            type="date"
            value={formData.fecha_inicio}
            onChange={(e) => setFormData({ ...formData, fecha_inicio: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Fecha de Fin</label>
          <input
            type="date"
            value={formData.fecha_fin}
            onChange={(e) => setFormData({ ...formData, fecha_fin: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Método de Pago</label>
          <select
            value={formData.metodo_pago}
            onChange={(e) => setFormData({ ...formData, metodo_pago: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="">Selecciona método</option>
            <option value="efectivo">Efectivo</option>
            <option value="transferencia">Transferencia</option>
            <option value="tarjeta">Tarjeta</option>
            <option value="cheque">Cheque</option>
            <option value="otro">Otro</option>
          </select>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Referencia</label>
          <input
            type="text"
            value={formData.referencia}
            onChange={(e) => setFormData({ ...formData, referencia: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="Número de referencia o comprobante"
          />
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Estado *</label>
          <select
            required
            value={formData.estado}
            onChange={(e) => setFormData({ ...formData, estado: e.target.value as any })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="completado">Completado</option>
            <option value="pendiente">Pendiente</option>
            <option value="cancelado">Cancelado</option>
          </select>
        </div>

        <div className="flex items-center gap-2">
          <input
            type="checkbox"
            id="incluir_iva"
            checked={formData.incluir_iva}
            onChange={(e) => setFormData({ ...formData, incluir_iva: e.target.checked })}
            className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
          />
          <label htmlFor="incluir_iva" className="text-sm font-medium">
            IVA / Impuesto
          </label>
        </div>

        <div>
          <label className="block text-sm font-medium mb-1">Notas</label>
          <textarea
            value={formData.notas}
            onChange={(e) => setFormData({ ...formData, notas: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            rows={3}
            placeholder="Notas adicionales sobre el pago"
          />
        </div>

        <div className="flex gap-4 pt-4">
          <button
            type="submit"
            disabled={guardando}
            className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 disabled:opacity-50 transition"
          >
            {guardando ? 'Guardando...' : 'Guardar Cambios'}
          </button>
          <button
            type="button"
            onClick={() => router.back()}
            className="bg-gray-300 text-gray-700 px-6 py-2 rounded hover:bg-gray-400 transition"
          >
            Cancelar
          </button>
        </div>
      </form>
    </div>
  )
}
